import 'package:flutter/material.dart';
import 'package:get/get.dart';
import 'package:nb_utils/nb_utils.dart';
import 'package:streamit_laravel/screens/home/components/slider_controller.dart';
import 'package:streamit_laravel/screens/home/shimmer_home.dart';
import 'package:streamit_laravel/screens/movie_list/banner_widget.dart';
import 'package:streamit_laravel/screens/video/shimmer_video.dart';
import 'package:streamit_laravel/screens/video/video_details_screen.dart';
import 'package:streamit_laravel/screens/video/video_list_controller.dart';
import 'package:streamit_laravel/utils/colors.dart';

import '../../components/app_scaffold.dart';
import '../../main.dart';
import '../../utils/animatedscroll_view_widget.dart';
import '../../utils/empty_error_state_widget.dart';

class VideoListScreen extends StatelessWidget {
  final String? type;
  final String? title;
  VideoListScreen({super.key, this.type = '', this.title});

  late final SliderController sliderController = Get.put(SliderController(type: type!));

  @override
  Widget build(BuildContext context) {
    return AppScaffoldNew(
        hasLeadingWidget: false,
        hideAppBar: true,
        scaffoldBackgroundColor: black,
        body: AnimatedListView(
            shrinkWrap: true,
            itemCount: 1,
            refreshIndicatorColor: appColorPrimary,
            padding: const EdgeInsets.only(bottom: 120),
            physics: const AlwaysScrollableScrollPhysics(),
            keyboardDismissBehavior: ScrollViewKeyboardDismissBehavior.onDrag,
            onNextPage: sliderController.onNextPage,
            onSwipeRefresh: () async {
              sliderController.getBanner(showLoader: true);
              return await sliderController.videoListController.getVideoList(showLoader: true);
            },
            itemBuilder: (context, index) {
              return Obx(() => SnapHelperWidget(
                  future: sliderController.getBannerList.value,
                  loadingWidget: const ShimmerHome(),
                  errorBuilder: (error) {
                    return SizedBox(
                      width: Get.width,
                      height: Get.height * 0.8,
                      child: NoDataWidget(
                        titleTextStyle: secondaryTextStyle(color: white),
                        subTitleTextStyle: primaryTextStyle(color: white),
                        title: error,
                        retryText: locale.value.reload,
                        imageWidget: const ErrorStateWidget(),
                        onRetry: () async {},
                      ).center(),
                    );
                  },
                  onSuccess: (res) {
                    return Column(
                      crossAxisAlignment: CrossAxisAlignment.start,
                      mainAxisSize: MainAxisSize.min,
                      children: [
                        BannerWidget(
                          sliderController: sliderController,
                        ),
                        Obx(()=> Text(
                          title.toString(),
                          style: primaryTextStyle(),
                        ).paddingOnly(left: 16).visible(!sliderController.isLoading.value)),

                        10.height,
                        Obx(() {
                          //  Data is loaded but list is empty → show NoDataWidget
                          if (sliderController.videoListController.videoList.isEmpty && !sliderController.videoListController.isLoading.value) {
                            return NoDataWidget(
                              titleTextStyle: boldTextStyle(color: white),
                              subTitleTextStyle: primaryTextStyle(color: white),
                              title: locale.value.noDataFound,
                              retryText: "",
                              imageWidget: const EmptyStateWidget(),
                            ).paddingSymmetric(horizontal: 16);
                          }

                          //  Data loaded and list is not empty → show list
                          return CustomAnimatedScrollView(
                            paddingLeft: Get.width * 0.04,
                            paddingRight: Get.width * 0.04,
                            paddingBottom: Get.height * 0.10,
                            spacing: Get.width * 0.03,
                            runSpacing: Get.height * 0.02,
                            posterHeight: 150,
                            posterWidth: Get.width * 0.286,
                            isHorizontalList: false,
                            isLoading: false,
                            isLastPage: sliderController.videoListController.isLastPage.value,
                            itemList: sliderController.videoListController.videoList,
                            onTap: (posterDet) {
                              Get.to(() => VideoDetailsScreen(), arguments: posterDet);
                            },
                            onNextPage: sliderController.onNextPage,
                            onSwipeRefresh: () async {
                              sliderController.videoListController.page(1);
                              return await sliderController.videoListController.getVideoList(showLoader: true);
                            },
                            isMovieList: true,
                          ).visible(!sliderController.isLoading.value);
                        })
                      ],
                    );
                  }));
            }));
  }
}
